const { firstVisibleTextFromLocators } = require('./_helpers');

// Точные селекторы под текущую вёрстку WB (h2 с цветом: primary = без клуба, accent = с клубом)
const WB_PRICE_PRIMARY = 'h2.mo-typography_color_primary';
const WB_PRICE_ACCENT = 'h2.mo-typography_color_accent';

const DEFAULT_PRICE_LOCATORS = [
  WB_PRICE_PRIMARY,
  'span.price-block__final-price',
  'ins.price-block__final-price',
  '[data-link="finalPrice"]',
  '[data-testid="price"]'
];

module.exports = {
  id: 'wildberries',
  match: (url) => /(^|\/\/)(www\.)?wildberries\.ru\//i.test(url),

  priceLocators: DEFAULT_PRICE_LOCATORS,

  async getRawPriceText({ page, timeoutMs }) {
    return firstVisibleTextFromLocators(page, this.priceLocators, timeoutMs);
  },

  /**
   * Раскрывает блок с ценой (клик по элементу с ценой, после чего появляются полные цены без клуба и с клубом).
   */
  async expandPriceBlock(page, timeoutMs = 10000) {
    const deadline = Date.now() + timeoutMs;
    const sel = WB_PRICE_ACCENT;
    try {
      const locator = page.locator(sel).first();
      await locator.waitFor({ state: 'visible', timeout: Math.min(5000, deadline - Date.now()) });
      await locator.click();
      await new Promise((r) => setTimeout(r, 1500));
    } catch (_) {
      // Элемент не найден или не кликабелен — продолжаем без клика
    }
  },

  /**
   * Возвращает две цены: обычная (h2.mo-typography_color_primary) и по WB Клубу (h2.mo-typography_color_accent).
   * Сначала кликаем по блоку цены, чтобы раскрыть полные цены, затем читаем их.
   */
  async getPrices({ page, selectorPrice, selectorClub, timeoutMs }) {
    await this.expandPriceBlock(page, Math.min(8000, timeoutMs || 20000));

    const priceLocators = selectorPrice
      ? [selectorPrice.trim(), WB_PRICE_PRIMARY, ...DEFAULT_PRICE_LOCATORS.filter((s) => s !== WB_PRICE_PRIMARY)]
      : DEFAULT_PRICE_LOCATORS;
    const clubLocators = selectorClub
      ? [selectorClub.trim(), WB_PRICE_ACCENT]
      : [WB_PRICE_ACCENT];

    const priceText = await firstVisibleTextFromLocators(page, priceLocators, timeoutMs);
    let clubPriceText = null;
    try {
      const accentCount = await page.locator(WB_PRICE_ACCENT).count();
      if (accentCount > 1) {
        const lastAccent = page.locator(WB_PRICE_ACCENT).last();
        await lastAccent.waitFor({ state: 'visible', timeout: 5000 });
        clubPriceText = (await lastAccent.innerText()).trim();
      } else {
        clubPriceText = await firstVisibleTextFromLocators(
          page,
          clubLocators,
          Math.min(10000, timeoutMs || 20000)
        );
      }
    } catch (_) {
      clubPriceText = null;
    }
    return { priceText: priceText || null, clubPriceText: clubPriceText || null };
  }
};

